"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_extra_1 = __importDefault(require("fs-extra"));
const path_1 = __importDefault(require("path"));
const EventHandler_1 = require("modloader64_api/EventHandler");
const NetworkHandler_1 = require("modloader64_api/NetworkHandler");
const modloader64_1 = require("./modloader64");
const LobbyVariable_1 = require("modloader64_api/LobbyVariable");
const CoreInjection_1 = require("modloader64_api/CoreInjection");
const GameShark_1 = require("modloader64_api/GameShark");
const PakFormat_1 = require("modloader64_api/PakFormat");
const crypto_1 = __importDefault(require("crypto"));
const GUITunnel_1 = require("modloader64_api/GUITunnel");
class pluginLoader {
    constructor(dirs, config, logger) {
        this.core_plugins = {};
        this.plugin_folders = [];
        this.plugins = [];
        this.selected_core = '';
        this.loaded_core = {};
        this.internalFrameCount = -1;
        this.plugin_directories = dirs;
        this.config = config;
        this.logger = logger;
        let cleanup = function () {
            fs_extra_1.default.readdirSync(process.cwd()).forEach((file) => {
                let parse = path_1.default.parse(file);
                if (parse.name.indexOf('ModLoader64_temp_') > -1) {
                    fs_extra_1.default.removeSync(file);
                }
            });
        };
        modloader64_1.internal_event_bus.on('SHUTDOWN_EVERYTHING', () => {
            cleanup();
        });
        cleanup();
    }
    verifySignature(file, key, sig) {
        const hasher = crypto_1.default.createHash('sha256');
        hasher.update(fs_extra_1.default.readFileSync(file));
        const digest = hasher.digest('hex');
        const publicKey = fs_extra_1.default.readFileSync(key);
        const verifier = crypto_1.default.createVerify('RSA-SHA256');
        verifier.update(digest);
        if (!fs_extra_1.default.existsSync(sig)) {
            return false;
        }
        const testSignature = verifier.verify(publicKey, fs_extra_1.default.readFileSync(sig).toString(), 'base64');
        return testSignature;
    }
    registerCorePlugin(name, core) {
        this.core_plugins[name] = core;
    }
    registerPlugin(plugin) {
        this.plugins.push(plugin);
    }
    processFolder(dir) {
        let parse = path_1.default.parse(dir);
        if (parse.ext === '.pak') {
            // Unpak first.
            let ndir = fs_extra_1.default.mkdtempSync('ModLoader64_temp_');
            let pakFile = new PakFormat_1.Pak(path_1.default.resolve(dir));
            pakFile.extractAll(ndir);
            dir = path_1.default.join(ndir, parse.name);
            let pub = path_1.default.join(dir, 'public_key.pem');
            if (fs_extra_1.default.existsSync(pub)) {
                if (!this.verifySignature(pakFile.fileName, path_1.default.resolve(pub), path_1.default.resolve(path_1.default.join(parse.dir, parse.name + '.sig')))) {
                    this.logger.error('Signature check failed for plugin ' + parse.name + '. Skipping.');
                    return;
                }
                else {
                    this.logger.info('Signature check for plugin ' + parse.name + ' passed.');
                }
            }
        }
        else if (parse.ext === '.sig') {
            return;
        }
        if (!fs_extra_1.default.lstatSync(path_1.default.resolve(dir)).isDirectory) {
            return;
        }
        let pkg_file = path_1.default.resolve(path_1.default.join(dir, 'package.json'));
        if (!fs_extra_1.default.existsSync(pkg_file)) {
            this.logger.error('Plugin ' + parse.name + ' is missing package.json. Skipping.');
            return;
        }
        let pkg = JSON.parse(fs_extra_1.default.readFileSync(pkg_file).toString());
        if (pkg.core !== this.selected_core && pkg.core !== '*') {
            this.logger.info('Plugin ' + pkg.name + ' does not belong to this core. Skipping.');
            return;
        }
        this.logger.info('--------------------');
        this.logger.info('plugin: ' + pkg.name);
        this.logger.info('version: ' + pkg.version);
        this.logger.info('author: ' + pkg.author);
        this.logger.info('additional credits: ' + pkg.credits);
        let file = path_1.default.resolve(path_1.default.join(dir, pkg.main));
        parse = path_1.default.parse(file);
        if (parse.ext.indexOf('js') > -1) {
            let p = require(file);
            let plugin = new p();
            plugin['ModLoader'] = {};
            plugin['ModLoader']['logger'] = this.logger.child({});
            plugin['ModLoader']['config'] = this.config;
            Object.defineProperty(plugin, 'pluginName', {
                value: parse.name,
                writable: false,
            });
            EventHandler_1.setupEventHandlers(plugin);
            NetworkHandler_1.setupNetworkHandlers(plugin);
            CoreInjection_1.setupCoreInject(plugin, this.loaded_core);
            LobbyVariable_1.setupLobbyVariable(plugin);
            Object.defineProperty(plugin, 'metadata', {
                value: pkg,
                writable: false,
            });
            this.registerPlugin(plugin);
            this.plugin_folders.push(parse.dir);
        }
    }
    loadPluginsConstruct(header, overrideCore = '') {
        // Start the core plugin.
        this.header = header;
        if (overrideCore !== '') {
            this.selected_core = overrideCore;
        }
        let core = this.core_plugins[this.selected_core];
        core['ModLoader'] = {};
        core['ModLoader']['logger'] = this.logger.child({});
        core['ModLoader']['config'] = this.config;
        this.loaded_core = core;
        Object.defineProperty(this.loaded_core, 'rom_header', {
            value: header,
            writable: false,
        });
        EventHandler_1.setupEventHandlers(this.loaded_core);
        NetworkHandler_1.setupNetworkHandlers(this.loaded_core);
        // Start external plugins.
        this.plugin_directories.forEach((dir) => {
            if (fs_extra_1.default.lstatSync(dir).isDirectory()) {
                let temp1 = path_1.default.resolve(path_1.default.join(dir));
                fs_extra_1.default.readdirSync(temp1).forEach((file) => {
                    let temp2 = path_1.default.join(temp1, file);
                    this.processFolder(temp2);
                });
            }
        });
        modloader64_1.internal_event_bus.on('onNetworkConnect', (evt) => {
            this.loaded_core.ModLoader.me = evt.me;
            this.plugins.forEach((plugin) => {
                plugin.ModLoader.me = evt.me;
            });
        });
    }
    loadPluginsPreInit(manager) {
        this.loaded_core.preinit();
        this.loaded_core.ModLoader.lobbyManager = manager;
        this.loaded_core.ModLoader.clientSide = NetworkHandler_1.ClientController;
        this.loaded_core.ModLoader.serverSide = NetworkHandler_1.ServerController;
        this.loaded_core.ModLoader.clientLobby = this.config.data['NetworkEngine.Client']['lobby'];
        this.plugins.forEach((plugin) => {
            plugin.ModLoader.lobbyManager = manager;
            plugin.ModLoader.clientSide = NetworkHandler_1.ClientController;
            plugin.ModLoader.serverSide = NetworkHandler_1.ServerController;
            plugin.ModLoader.clientLobby = this.config.data['NetworkEngine.Client']['lobby'];
            plugin.preinit();
        });
    }
    loadPluginsInit(me, iconsole, net) {
        this.loaded_core.ModLoader.me = me;
        this.loaded_core.init();
        this.plugins.forEach((plugin) => {
            plugin.ModLoader.me = me;
            plugin.init();
        });
        this.onTickHandle = (frame) => {
            this.internalFrameCount = frame;
            this.loaded_core.onTick();
            this.plugins.forEach((plugin) => {
                plugin.onTick();
            });
            net.onTick();
        };
        iconsole.setFrameCallback(this.onTickHandle);
    }
    loadPluginsPostinit(emulator, iconsole) {
        let mainConfig = this.config.registerConfigCategory('ModLoader64');
        let utils = emulator;
        this.loaded_core.ModLoader.emulator = emulator;
        this.loaded_core.ModLoader.utils = utils;
        this.loaded_core.ModLoader.savestates = emulator;
        this.loaded_core.ModLoader.gui = new GUITunnel_1.GUIAPI('core', this.loaded_core);
        this.loaded_core.postinit();
        this.plugins.forEach((plugin) => {
            plugin.ModLoader.emulator = emulator;
            plugin.ModLoader.utils = utils;
            plugin.ModLoader.gui = new GUITunnel_1.GUIAPI(plugin.pluginName, plugin);
            plugin.ModLoader.savestates = emulator;
            plugin.postinit();
            if (mainConfig.isClient) {
                EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_PLUGIN_READY, plugin);
            }
            if (mainConfig.isServer) {
                EventHandler_1.bus.emit(EventHandler_1.EventsServer.ON_PLUGIN_READY, plugin);
            }
        });
        this.onFakeFrameHandler = setInterval(() => {
            if (this.internalFrameCount >= 0) {
                clearInterval(this.onFakeFrameHandler);
                iconsole.pauseEmulator();
                let gameshark = new GameShark_1.GameShark(this.logger, emulator);
                this.plugin_folders.forEach((dir) => {
                    let test = path_1.default.join(dir, 'payloads', this.header.country_code + this.header.revision.toString());
                    if (fs_extra_1.default.existsSync(test)) {
                        if (fs_extra_1.default.lstatSync(test).isDirectory) {
                            fs_extra_1.default.readdirSync(test).forEach((payload) => {
                                gameshark.read(path_1.default.resolve(path_1.default.join(test, payload)));
                            });
                        }
                    }
                });
                EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_INJECT_FINISHED, {});
                iconsole.finishInjects();
                iconsole.resumeEmulator();
            }
        }, 50);
        iconsole.hookFrameCallback();
    }
}
exports.default = pluginLoader;
//# sourceMappingURL=pluginLoader.js.map