"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const config_1 = __importDefault(require("./config"));
const pluginLoader_1 = __importDefault(require("./pluginLoader"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const IModLoaderAPI_1 = require("modloader64_api/IModLoaderAPI");
const NetworkEngine_1 = __importDefault(require("./NetworkEngine"));
const N64_1 = __importDefault(require("./consoles/N64"));
const FakeMupen_1 = require("./consoles/FakeMupen");
const EventHandler_1 = require("modloader64_api/EventHandler");
const GUITunnel_1 = require("modloader64_api/GUITunnel");
const SUPPORTED_CONSOLES = ['N64'];
exports.internal_event_bus = new EventHandler_1.EventBus();
class ModLoader64 {
    constructor(logger) {
        this.config = new config_1.default(process.cwd() + '/ModLoader64-config.json');
        this.data = this.config.registerConfigCategory('ModLoader64');
        this.rom_folder = './roms';
        this.mods_folder = './mods';
        this.done = false;
        if (!fs_1.default.existsSync(this.rom_folder)) {
            fs_1.default.mkdirSync(this.rom_folder);
        }
        if (!fs_1.default.existsSync(this.mods_folder)) {
            fs_1.default.mkdirSync(this.mods_folder);
        }
        this.roms = fs_1.default.readdirSync(this.rom_folder);
        this.logger = logger;
        this.plugins = new pluginLoader_1.default([path_1.default.resolve(path_1.default.join(process.cwd(), 'mods'))], this.config, this.logger.child({}));
        this.Server = new NetworkEngine_1.default.Server(this.logger.child({}), this.config);
        this.Client = new NetworkEngine_1.default.Client(this.logger.child({}), this.config);
        if (process.platform === 'win32') {
            let rl = require('readline').createInterface({
                input: process.stdin,
                output: process.stdout,
            });
            rl.on('SIGINT', function () {
                // @ts-ignore
                process.emit('SIGINT');
            });
        }
        process.on('SIGINT', function () {
            //graceful shutdown
            exports.internal_event_bus.emit('SHUTDOWN_EVERYTHING', {});
            process.exit(0);
        });
        process.on('message', (msg) => {
            let packet = JSON.parse(msg);
            EventHandler_1.bus.emit(packet.id, packet);
        });
    }
    start() {
        this.tunnel = new GUITunnel_1.GUITunnel(process, 'internal_event_bus', this);
        let ofn = exports.internal_event_bus.emit.bind(exports.internal_event_bus);
        ((inst) => {
            exports.internal_event_bus.emit = function (event, ...args) {
                inst.tunnel.send(event, args);
                return ofn(event, args);
            };
        })(this);
        exports.internal_event_bus.on('SHUTDOWN_EVERYTHING', () => {
            this.emulator.stopEmulator();
        });
        this.preinit();
    }
    preinit() {
        // Set up config.
        this.config.setData('ModLoader64', 'rom', 'Legend of Zelda, The - Ocarina of Time (U) (V1.0) [!].z64');
        this.config.setData('ModLoader64', 'patch', '');
        this.config.setData('ModLoader64', 'isServer', true);
        this.config.setData('ModLoader64', 'isClient', true);
        this.config.setData('ModLoader64', 'supportedConsoles', SUPPORTED_CONSOLES, true);
        this.config.setData('ModLoader64', 'selectedConsole', 'N64');
        this.rom_path = path_1.default.resolve(path_1.default.join(this.rom_folder, this.data['rom']));
        fs_1.default.readdirSync(path_1.default.resolve(path_1.default.join(__dirname, '/cores'))).forEach(file => {
            let f = path_1.default.join(__dirname, '/cores', file);
            if (!fs_1.default.lstatSync(f).isDirectory()) {
                let parse = path_1.default.parse(file);
                if (parse.ext === '.js') {
                    let p = require(f)[parse.name];
                    this.plugins.registerCorePlugin(parse.name, new p());
                    this.logger.info('Auto-wiring core: ' + parse.name);
                }
            }
        });
        if (this.data.isServer) {
            switch (this.data.selectedConsole) {
                case 'N64': {
                    this.emulator = new FakeMupen_1.FakeMupen(this.rom_path);
                    break;
                }
            }
        }
        if (this.data.isClient) {
            switch (this.data.selectedConsole) {
                case 'N64': {
                    this.emulator = new N64_1.default(this.rom_path, this.logger.child({}));
                    break;
                }
            }
        }
        exports.internal_event_bus.emit('preinit_done', {});
        this.init();
    }
    init() {
        let loaded_rom_header;
        if (fs_1.default.existsSync(this.rom_path)) {
            this.logger.info('Parsing rom header...');
            loaded_rom_header = this.emulator.getRomHeader();
            let core_match = null;
            let core_key = '';
            Object.keys(this.plugins.core_plugins).forEach((key) => {
                if (loaded_rom_header.name === this.plugins.core_plugins[key].header) {
                    core_match = this.plugins.core_plugins[key];
                    core_key = key;
                }
            });
            if (core_match !== null) {
                this.logger.info('Auto-selected core: ' + core_key);
                this.plugins.selected_core = core_key;
            }
            else {
                this.logger.error('Failed to find a compatible core for the selected rom!');
            }
            // Load the plugins
            this.plugins.loadPluginsConstruct(loaded_rom_header);
            EventHandler_1.bus.emit(IModLoaderAPI_1.ModLoaderEvents.ON_ROM_PATH, this.rom_path);
            EventHandler_1.bus.emit(IModLoaderAPI_1.ModLoaderEvents.ON_ROM_HEADER_PARSED, loaded_rom_header);
        }
        this.plugins.loadPluginsPreInit(this.Server);
        exports.internal_event_bus.emit('onPreInitDone', {});
        // Set up networking.
        exports.internal_event_bus.on('onNetworkConnect', (evt) => {
            this.postinit(evt);
        });
        (() => {
            if (this.data.isServer) {
                this.Server.setup();
            }
            if (this.data.isClient) {
                this.Client.setup();
            }
        })();
        exports.internal_event_bus.emit('onInitDone', {});
    }
    postinit(result) {
        if (this.done) {
            return;
        }
        if (fs_1.default.existsSync(this.rom_path)) {
            this.plugins.loadPluginsInit(result[0].me, this.emulator, this.Client);
            this.logger.info('Setting up Mupen...');
            let instance = this;
            let mupen;
            let load_mupen = new Promise(function (resolve, reject) {
                mupen = instance.emulator.startEmulator(() => {
                    let p = result[0].patch;
                    let rom_data = instance.emulator.getLoadedRom();
                    if (p.byteLength > 1) {
                        let BPS = require('./BPS');
                        let _BPS = new BPS();
                        rom_data = _BPS.tryPatch(rom_data, p);
                    }
                    EventHandler_1.bus.emit(IModLoaderAPI_1.ModLoaderEvents.ON_ROM_PATCHED, { rom: rom_data });
                    return rom_data;
                });
                while (!instance.emulator.isEmulatorReady()) { }
                exports.internal_event_bus.emit('emulator_started', {});
                resolve();
            });
            load_mupen.then(function () {
                instance.logger.info('Finishing plugin init...');
                instance.plugins.loadPluginsPostinit(mupen, instance.emulator);
                exports.internal_event_bus.emit('onPostInitDone', {});
                instance.done = true;
                // Detect if the user closed Mupen. Exit with code 1.
                setInterval(() => {
                    if (!instance.emulator.isEmulatorReady()) {
                        exports.internal_event_bus.emit('SHUTDOWN_EVERYTHING', {});
                        process.exit(1);
                    }
                }, 1000);
            });
        }
    }
}
exports.default = ModLoader64;
//# sourceMappingURL=modloader64.js.map