"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const JSONTemplate_1 = require("modloader64_api/JSONTemplate");
const BootsEquipment_1 = require("./BootsEquipment");
const TunicsEquipment_1 = require("./TunicsEquipment");
const ShieldsEquipment_1 = require("./ShieldsEquipment");
const SwordsEquipment_1 = require("./SwordsEquipment");
const Inventory_1 = require("./Inventory");
const QuestStatus_1 = require("./QuestStatus");
const ZeldaString_1 = require("modloader64_api/OOT/ZeldaString");
class SaveContext extends JSONTemplate_1.JSONTemplate {
    constructor(emu, log) {
        super();
        this.instance = global.ModLoader.save_context;
        this.entrance_index_addr = this.instance + 0x0000;
        this.age_addr = this.instance + 0x0004;
        this.cutscene_number_addr = this.instance + 0x000a;
        this.world_time_addr = this.instance + 0x000c;
        this.world_night_addr = this.instance + 0x0010;
        this.zeldaz_addr = this.instance + 0x001c;
        this.death_addr = this.instance + 0x0022;
        this.player_name_addr = this.instance + 0x0024;
        this.dd_flag_addr = this.instance + 0x002c;
        this.heart_container_addr = this.instance + 0x002e;
        this.health_addr = this.instance + 0x0030;
        this.magic_meter_size_addr = this.instance + 0x0032;
        this.magic_current_addr = this.instance + 0x0033;
        this.magic_limit_addr = this.instance + 0x13f4;
        this.magic_flag_1_addr = this.instance + 0x003a;
        this.magic_flag_2_addr = this.instance + 0x003c;
        this.rupees_address = this.instance + 0x0034;
        this.navi_timer_addr = this.instance + 0x0038;
        this.checksum_addr = this.instance + 0x1352;
        this.magic_beans_addr = this.instance + 0x009b;
        this.scene_data_addr = this.instance + 0x00d4;
        this.event_data_addr = this.instance + 0x0ed4;
        this.item_flag_addr = this.instance + 0x0ef0;
        this.inf_table_addr = this.instance + 0x0ef8;
        this.jsonFields = [
            'entrance_index',
            'cutscene_number',
            'world_time',
            'world_night_flag',
            'zeldaz_string',
            'death_counter',
            'player_name',
            'dd_flag',
            'heart_containers',
            'health',
            'magic_meter_size',
            'magic_current',
            'rupee_count',
            'navi_timer',
            'checksum',
            'age',
            'swords',
            'shields',
            'tunics',
            'boots',
            'inventory',
            'questStatus',
            'magic_beans_purchased',
        ];
        this.emulator = emu;
        this.swords = new SwordsEquipment_1.SwordsEquipment(emu);
        this.shields = new ShieldsEquipment_1.ShieldsEquipment(emu);
        this.tunics = new TunicsEquipment_1.TunicsEquipment(emu);
        this.boots = new BootsEquipment_1.BootsEquipment(emu);
        this.inventory = new Inventory_1.Inventory(emu, log);
        this.questStatus = new QuestStatus_1.QuestStatus(emu);
    }
    // https://wiki.cloudmodding.com/oot/Entrance_Table
    // https://wiki.cloudmodding.com/oot/Entrance_Table_(Data)
    get entrance_index() {
        return this.emulator.rdramRead32(this.entrance_index_addr);
    }
    set entrance_index(index) {
        this.emulator.rdramWrite32(this.entrance_index_addr, index);
    }
    get cutscene_number() {
        return this.emulator.rdramRead16(this.cutscene_number_addr);
    }
    set cutscene_number(index) {
        this.emulator.rdramWrite16(this.cutscene_number_addr, index);
    }
    get world_time() {
        return this.emulator.rdramRead16(this.world_time_addr);
    }
    set world_time(time) {
        this.emulator.rdramWrite16(this.world_time_addr, time);
    }
    get world_night_flag() {
        return this.emulator.rdramRead32(this.world_night_addr) === 1;
    }
    set world_night_flag(bool) {
        this.emulator.rdramWrite32(this.world_night_addr, (function (bool) {
            return bool ? 1 : 0;
        })(bool));
    }
    get zeldaz_string() {
        return this.emulator.rdramReadBuffer(this.zeldaz_addr, 6).toString('ascii');
    }
    get death_counter() {
        return this.emulator.rdramRead16(this.death_addr);
    }
    set death_counter(deaths) {
        this.emulator.rdramWrite16(this.death_addr, deaths);
    }
    get player_name() {
        return ZeldaString_1.zeldaString.decode(this.emulator.rdramReadBuffer(this.player_name_addr, 8));
    }
    // Will always be false normally.
    get dd_flag() {
        return this.emulator.rdramRead16(this.dd_flag_addr) === 1;
    }
    set dd_flag(bool) {
        this.emulator.rdramWrite16(this.dd_flag_addr, (function (bool) {
            return bool ? 1 : 0;
        })(bool));
    }
    get heart_containers() {
        return this.emulator.rdramRead16(this.heart_container_addr) / 0x10;
    }
    set heart_containers(num) {
        this.emulator.rdramWrite16(this.heart_container_addr, num * 0x10);
    }
    get health() {
        return this.emulator.rdramRead16(this.health_addr) / 0x10;
    }
    set health(hearts) {
        this.emulator.rdramWrite16(this.health_addr, hearts * 0x10);
    }
    get magic_meter_size() {
        return this.emulator.rdramRead8(this.magic_meter_size_addr);
    }
    // Several things need to be set in order for magic to function properly.
    set magic_meter_size(size) {
        this.emulator.rdramWrite8(this.magic_meter_size_addr, size);
        switch (size) {
            case 0 /* NONE */: {
                this.emulator.rdramWrite8(this.magic_limit_addr, 0 /* NONE */);
                this.emulator.rdramWrite8(this.magic_flag_1_addr, 0);
                this.emulator.rdramWrite8(this.magic_flag_2_addr, 0);
                this.magic_current = 0 /* NONE */;
                break;
            }
            case 1 /* NORMAL */: {
                this.emulator.rdramWrite8(this.magic_limit_addr, 48 /* NORMAL */);
                this.emulator.rdramWrite8(this.magic_flag_1_addr, 1);
                this.emulator.rdramWrite8(this.magic_flag_2_addr, 0);
                this.magic_current = 48 /* NORMAL */;
                break;
            }
            case 2 /* EXTENDED */: {
                this.emulator.rdramWrite8(this.magic_limit_addr, 96 /* EXTENDED */);
                this.emulator.rdramWrite8(this.magic_flag_1_addr, 1);
                this.emulator.rdramWrite8(this.magic_flag_2_addr, 1);
                this.magic_current = 96 /* EXTENDED */;
                break;
            }
        }
    }
    get magic_current() {
        return this.emulator.rdramRead8(this.magic_current_addr);
    }
    // Failsafe to keep people from setting the magic amount over the cap.
    set magic_current(amount) {
        this.emulator.rdramWrite8(this.magic_current, (function (amt) {
            return amt > 96 /* EXTENDED */ ? 96 /* EXTENDED */ : amt;
        })(amount));
    }
    get rupee_count() {
        return this.emulator.rdramRead16(this.rupees_address);
    }
    set rupee_count(dosh) {
        this.emulator.rdramWrite16(this.rupees_address, dosh);
    }
    get navi_timer() {
        return this.emulator.rdramRead16(this.navi_timer_addr);
    }
    set navi_timer(time) {
        this.emulator.rdramWrite16(this.navi_timer_addr, time);
    }
    get checksum() {
        return this.emulator.rdramRead16(this.checksum_addr);
    }
    get age() {
        return this.emulator.rdramRead32(this.age_addr);
    }
    get magic_beans_purchased() {
        return this.emulator.rdramRead8(this.magic_beans_addr);
    }
    set magic_beans_purchased(amt) {
        this.emulator.rdramWrite8(this.magic_beans_addr, amt);
    }
    get permSceneData() {
        return this.emulator.rdramReadBuffer(this.scene_data_addr, 0xb0c);
    }
    set permSceneData(buf) {
        this.emulator.rdramWriteBuffer(this.scene_data_addr, buf);
    }
    get eventFlags() {
        return this.emulator.rdramReadBuffer(this.event_data_addr, 0x1c);
    }
    set eventFlags(buf) {
        this.emulator.rdramWriteBuffer(this.event_data_addr, buf);
    }
    get itemFlags() {
        return this.emulator.rdramReadBuffer(this.item_flag_addr, 0x8);
    }
    set itemFlags(buf) {
        this.emulator.rdramWriteBuffer(this.item_flag_addr, buf);
    }
    get infTable() {
        return this.emulator.rdramReadBuffer(this.inf_table_addr, 0x3c);
    }
    set infTable(buf) {
        this.emulator.rdramWriteBuffer(this.inf_table_addr, buf);
    }
}
exports.SaveContext = SaveContext;
//# sourceMappingURL=SaveContext.js.map