"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Actor_1 = require("./Actor");
const crypto_1 = __importDefault(require("crypto"));
const EventHandler_1 = require("modloader64_api/EventHandler");
const OOTAPI_1 = require("modloader64_api/OOT/OOTAPI");
class ActorManager {
    constructor(emulator, logger, helper) {
        this.actor_array_addr = 0x001c30;
        this.ringbuffer_start_addr = 0x600000 + 0x1e0;
        this.ringbuffer_index_addr = 0x600000 + 0x11e0;
        this.ringbuffer_entry_size = 0x10;
        this.ringbuffer_max = 0x100;
        this.actor_next_offset = 0x124;
        this.actors_pointers_this_frame = new Array();
        this.actors_this_frame = new Map();
        this.actors_without_rom_ids_this_frame = new Map();
        this.emulator = emulator;
        this.logger = logger;
        this.helper = helper;
        for (let i = 0; i < 12; i++) {
            this.actors_this_frame.set(i, new Array());
        }
    }
    createIActorFromPointer(pointer) {
        return new Actor_1.ActorBase(this.emulator, pointer);
    }
    onTick() {
        this.actors_pointers_this_frame.length = 0;
        this.actors_without_rom_ids_this_frame.clear();
        if (!this.helper.isLinkEnteringLoadingZone()) {
            for (let i = 0; i < 12 * 8; i += 8) {
                let count = this.emulator.rdramReadPtr32(global.ModLoader.global_context_pointer, this.actor_array_addr + i);
                let ptr = this.emulator.dereferencePointer(global.ModLoader.global_context_pointer);
                if (count > 0) {
                    let pointer = this.emulator.dereferencePointer(ptr + this.actor_array_addr + (i + 4));
                    this.actors_pointers_this_frame.push(pointer);
                    let next = this.emulator.dereferencePointer(pointer + this.actor_next_offset);
                    while (next > 0) {
                        this.actors_pointers_this_frame.push(next);
                        next = this.emulator.dereferencePointer(next + this.actor_next_offset);
                    }
                }
            }
        }
        for (let i = 0; i < this.ringbuffer_max * this.ringbuffer_entry_size; i += this.ringbuffer_entry_size) {
            // Get all the entry data.
            let category = this.emulator.rdramRead32(this.ringbuffer_start_addr + i);
            // Break asap if we hit a zero. This means there is no further data that needs our attention this frame.
            if (category === 0) {
                break;
            }
            let addr = this.emulator.dereferencePointer(this.ringbuffer_start_addr + i + 4);
            let rom = this.emulator.rdramRead32(this.ringbuffer_start_addr + i + 8);
            let n = 0;
            if (addr > 0) {
                if (this.actors_pointers_this_frame.indexOf(addr) > -1) {
                    let actor = new Actor_1.ActorBase(this.emulator, addr);
                    if (!this.actors_without_rom_ids_this_frame.has(actor.actorID)) {
                        this.actors_without_rom_ids_this_frame.set(actor.actorID, 0);
                    }
                    n = this.actors_without_rom_ids_this_frame.get(actor.actorID);
                    n++;
                    this.actors_without_rom_ids_this_frame.set(actor.actorID, n);
                    let uuid = crypto_1.default
                        .createHash('md5')
                        .update(Buffer.from(rom.toString(16) +
                        actor.actorID.toString(16) +
                        actor.room.toString(16) +
                        n.toString(16)))
                        .digest('hex');
                    actor.actorUUID = uuid;
                    this.actors_this_frame.get(actor.actorType).push(actor);
                    EventHandler_1.bus.emit(OOTAPI_1.OotEvents.ON_ACTOR_SPAWN, actor);
                }
            }
            // Clear the entry when we're done with it.
            this.emulator.rdramWrite32(this.ringbuffer_start_addr + i, 0);
            this.emulator.rdramWrite32(this.ringbuffer_start_addr + i + 4, 0);
            this.emulator.rdramWrite32(this.ringbuffer_start_addr + i + 8, 0);
            this.emulator.rdramWrite32(this.ringbuffer_start_addr + i + 12, 0);
        }
        this.actors_this_frame.forEach((value, key) => {
            for (let i = 0; i < value.length; i++) {
                if (this.actors_pointers_this_frame.indexOf(value[i].instance) === -1) {
                    value[i].exists = false;
                    let removed = value.splice(i, 1)[0];
                    EventHandler_1.bus.emit(OOTAPI_1.OotEvents.ON_ACTOR_DESPAWN, removed);
                }
            }
        });
        // Clear the ring buffer index so it snaps back to the top.
        this.emulator.rdramWrite32(this.ringbuffer_index_addr, 0);
    }
}
exports.ActorManager = ActorManager;
//# sourceMappingURL=ActorManager.js.map