"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const EventHandler_1 = require("modloader64_api/EventHandler");
const NetworkHandler_1 = require("modloader64_api/NetworkHandler");
const crypto_1 = __importDefault(require("crypto"));
const ModLoaderDefaultImpls_1 = require("modloader64_api/ModLoaderDefaultImpls");
const fs_1 = __importDefault(require("fs"));
const uuid_1 = __importDefault(require("uuid"));
const modloader64_1 = require("./modloader64");
const zlib_1 = __importDefault(require("zlib"));
const dgram_1 = __importDefault(require("dgram"));
const path_1 = __importDefault(require("path"));
let natUpnp = require('nat-upnp');
let natUpnp_client = natUpnp.createClient();
let portfinder = require('portfinder');
class Version {
    constructor(major, minor, build) {
        this.major = parseInt(major);
        this.minor = parseInt(minor);
        this.build = parseInt(build);
    }
    match(v) {
        return this.major === v.major && this.minor === this.minor;
    }
}
class LobbyStorage {
    constructor(config, owner) {
        this.config = config;
        this.owner = owner;
        this.data = {};
    }
}
class LobbyJoin {
    constructor(lobbyData, player) {
        this.lobbyData = lobbyData;
        this.player = player;
    }
}
class FakeNetworkPlayer {
    constructor() {
        this.nickname = 'FakeNetworkPlayer';
        this.uuid = uuid_1.default.v4();
    }
    isSamePlayer(compare) {
        return compare.nickname === this.nickname && compare.uuid === this.uuid;
    }
}
var NetworkEngine;
(function (NetworkEngine) {
    class Server {
        constructor(logger, config) {
            this.encrypt = require('socket.io-encrypt');
            this.lobbyVariables = new Array();
            this.currently_processing_lobby = '';
            this.fakePlayer = new FakeNetworkPlayer();
            this.udpServer = dgram_1.default.createSocket('udp4');
            this.udpPort = -1;
            this.logger = logger;
            this.masterConfig = config;
            this.config = config.registerConfigCategory('NetworkEngine.Server');
            config.setData('NetworkEngine.Server', 'port', 8082);
            let temp = global.ModLoader.version.split('.');
            this.version = new Version(temp[0], temp[1], temp[2]);
            this.modLoaderconfig = this.masterConfig.registerConfigCategory('ModLoader64');
            EventHandler_1.bus.on('setupLobbyVariable', evt => {
                this.lobbyVariables.push(evt);
            });
        }
        getLobbies() {
            return this.io.sockets.adapter.rooms;
        }
        doesLobbyExist(Lobby) {
            return this.getLobbies().hasOwnProperty(Lobby);
        }
        createLobbyStorage(ld, owner) {
            this.getLobbies()[ld.name]['ModLoader64'] = new LobbyStorage(ld, owner);
            return this.getLobbies()[ld.name]['ModLoader64'];
        }
        getLobbyStorage(name) {
            try {
                return this.getLobbies()[name].ModLoader64;
            }
            catch (err) { }
            //@ts-ignore
            return null;
        }
        sendToTarget(target, internalChannel, packet) {
            this.io.to(target).emit(internalChannel, packet);
        }
        setup() {
            const server = require('http').createServer();
            this.io = require('socket.io')(server);
            server.listen(this.config.port);
            this.io.use(this.encrypt(global.ModLoader.version));
            modloader64_1.internal_event_bus.on('SHUTDOWN_EVERYTHING', () => {
                this.logger.info('SHUTDOWN DETECTED.');
            });
            this.logger.info('NetworkEngine.Server set up on port ' + this.config.port + '.');
            if (!this.modLoaderconfig.isClient) {
                modloader64_1.internal_event_bus.emit('onNetworkConnect', {
                    me: this.fakePlayer,
                    patch: Buffer.alloc(1),
                });
            }
            (function (inst) {
                natUpnp_client.portMapping({
                    public: inst.config.port,
                    private: inst.config.port,
                    ttl: 10,
                }, function (err) {
                    if (err) {
                        inst.logger.error("Didn't open port for TCP server.");
                    }
                    else {
                        inst.logger.info('Opened port for TCP server.');
                    }
                });
                portfinder.getPort((err, port) => {
                    natUpnp_client.portMapping({
                        public: port,
                        private: port,
                        ttl: 10,
                    }, function (err) {
                        if (err) {
                            inst.logger.error("Didn't open port for UDP server.");
                        }
                        else {
                            inst.logger.info('Opened port for UDP server.');
                        }
                    });
                    inst.udpPort = port;
                    inst.udpServer.bind(port);
                    NetworkHandler_1.NetworkSendBusServer.addListener('msg', (data) => {
                        if (data.lobby === undefined) {
                            data.lobby = inst.currently_processing_lobby;
                        }
                        if (data.player === undefined) {
                            data.player = inst.fakePlayer;
                        }
                        inst.sendToTarget(data.lobby, 'msg', data);
                    });
                    NetworkHandler_1.NetworkSendBusServer.addListener('toPlayer', (data) => {
                        inst.sendToTarget(data.player.uuid, 'msg', data.packet);
                    });
                    inst.io.on('connection', function (socket) {
                        inst.logger.info('Client ' + socket.id + ' connected.');
                        inst.sendToTarget(socket.id, 'uuid', { uuid: socket.id });
                        socket.on('version', function (data) {
                            let parse = data.split('.');
                            let v = new Version(parse[0], parse[1], parse[2]);
                            if (inst.version.match(v)) {
                                inst.sendToTarget(socket.id, 'versionGood', {
                                    client: v,
                                    server: inst.version,
                                    uuid: socket.id,
                                });
                            }
                            else {
                                inst.sendToTarget(socket.id, 'versionBad', {
                                    client: v,
                                    server: inst.version,
                                });
                                setTimeout(function () {
                                    socket.disconnect();
                                }, 1000);
                            }
                        });
                        socket.on('LobbyRequest', function (lj) {
                            if (inst.doesLobbyExist(lj.lobbyData.name)) {
                                // Lobby already exists.
                                let storage = inst.getLobbyStorage(lj.lobbyData.name);
                                if (storage.config.key === lj.lobbyData.key) {
                                    socket.join(storage.config.name);
                                    EventHandler_1.bus.emit(EventHandler_1.EventsServer.ON_LOBBY_JOIN, new EventHandler_1.EventServerJoined(lj.player, lj.lobbyData.name));
                                    //@ts-ignore
                                    socket['ModLoader64'] = {
                                        lobby: storage.config.name,
                                        player: lj.player,
                                    };
                                    inst.sendToTarget(socket.id, 'LobbyReady', {
                                        storage: storage.config,
                                        udp: inst.udpPort,
                                    });
                                    inst.sendToTarget(lj.lobbyData.name, 'playerJoined', lj.player);
                                }
                                else {
                                    inst.sendToTarget(socket.id, 'LobbyDenied_BadPassword', lj);
                                }
                            }
                            else {
                                // Lobby does not exist.
                                inst.logger.info('Creating lobby ' + lj.lobbyData.name + '.');
                                socket.join(lj.lobbyData.name);
                                let storage = inst.createLobbyStorage(lj.lobbyData, socket.id);
                                inst.lobbyVariables.forEach((value, index, array) => {
                                    storage.data[value.objectKey] = {};
                                    storage.data[value.objectKey][value.fieldName] = value.cloneTemplate();
                                });
                                EventHandler_1.bus.emit(EventHandler_1.EventsServer.ON_LOBBY_CREATE, storage);
                                EventHandler_1.bus.emit(EventHandler_1.EventsServer.ON_LOBBY_JOIN, new EventHandler_1.EventServerJoined(lj.player, lj.lobbyData.name));
                                //@ts-ignore
                                socket['ModLoader64'] = {
                                    lobby: storage.config.name,
                                    player: lj.player,
                                };
                                inst.sendToTarget(socket.id, 'LobbyReady', {
                                    storage: storage.config,
                                    udp: inst.udpPort,
                                });
                                inst.sendToTarget(lj.lobbyData.name, 'playerJoined', lj.player);
                            }
                        });
                        socket.on('playerJoined_reply', function (data) {
                            inst.sendToTarget(data.dest.uuid, 'playerJoined_bounce', data.player);
                        });
                        socket.on('msg', function (data) {
                            inst.currently_processing_lobby = data.lobby;
                            inst.lobbyVariables.forEach((value, index, array) => {
                                value.setField(inst.getLobbyStorage(data.lobby).data[value.objectKey][value.fieldName]);
                            });
                            NetworkHandler_1.NetworkBusServer.emit(data.packet_id, data);
                            NetworkHandler_1.NetworkChannelBusServer.emit(data.channel, data);
                            if (data.forward) {
                                socket.to(data.lobby).emit('msg', data);
                            }
                            inst.currently_processing_lobby = '';
                        });
                        socket.on('toSpecificPlayer', function (data) {
                            inst.sendToTarget(data.player.uuid, 'msg', data.packet);
                        });
                        socket.on('disconnect', () => {
                            //@ts-ignore
                            let ML = socket.ModLoader64;
                            if (ML === undefined)
                                return;
                            EventHandler_1.bus.emit(EventHandler_1.EventsServer.ON_LOBBY_LEAVE, new EventHandler_1.EventServerLeft(ML.player, ML.lobby));
                            inst.sendToTarget(ML.lobby, 'left', ML.player);
                        });
                    });
                    inst.udpServer.on('error', (err) => {
                        inst.logger.error(`server error:\n${err.stack}`);
                        inst.udpServer.close();
                    });
                    inst.udpServer.on('message', (msg, rinfo) => {
                        let data = JSON.parse(msg);
                        if (data.packet_id === 'UDPTestPacket') {
                            let reply = JSON.parse(JSON.stringify(data));
                            reply.player = inst.fakePlayer;
                            inst.sendToTarget(data.player.uuid, 'udpTest', reply);
                            return;
                        }
                        inst.currently_processing_lobby = data.lobby;
                        inst.lobbyVariables.forEach((value, index, array) => {
                            if (inst.getLobbyStorage(data.lobby) === null) {
                                return;
                            }
                            value.setField(inst.getLobbyStorage(data.lobby).data[value.objectKey][value.fieldName]);
                        });
                        NetworkHandler_1.NetworkBusServer.emit(data.packet_id, data);
                        NetworkHandler_1.NetworkChannelBusServer.emit(data.channel, data);
                        if (data.forward) {
                            Object.keys(inst.io.sockets.adapter.rooms[data.lobby].sockets).forEach((key) => {
                                if (key !== data.player.uuid) {
                                    inst.sendToTarget(key, 'msg', data);
                                }
                            });
                        }
                        inst.currently_processing_lobby = '';
                    });
                    inst.udpServer.on('listening', () => {
                        const address = inst.udpServer.address();
                        inst.logger.info(`UDP socket listening ${address.address}:${address.port}`);
                    });
                });
            })(this);
        }
    }
    NetworkEngine.Server = Server;
    class UDPTestPacket extends ModLoaderDefaultImpls_1.UDPPacket {
        constructor() {
            super('UDPTestPacket', 'ModLoader64', false);
        }
    }
    class Client {
        constructor(logger, config) {
            this.io = require('socket.io-client');
            this.socket = {};
            this.encrypt = require('socket.io-encrypt');
            this.udpClient = dgram_1.default.createSocket('udp4');
            this.serverUDPPort = -1;
            this.isUDPEnabled = false;
            this.packetBuffer = new Array();
            this.logger = logger;
            this.config = config.registerConfigCategory('NetworkEngine.Client');
            config.setData('NetworkEngine.Client', 'ip', '127.0.0.1');
            config.setData('NetworkEngine.Client', 'port', 8082);
            config.setData('NetworkEngine.Client', 'lobby', require('human-readable-ids').hri.random());
            config.setData('NetworkEngine.Client', 'nickname', 'Player');
            config.setData('NetworkEngine.Client', 'password', '');
            this.masterConfig = config;
            this.modLoaderconfig = this.masterConfig.registerConfigCategory('ModLoader64');
            modloader64_1.internal_event_bus.on('SHUTDOWN_EVERYTHING', () => {
                this.socket.removeAllListeners();
                this.socket.disconnect();
            });
        }
        onTick() {
            while (this.packetBuffer.length > 0) {
                let data = this.packetBuffer.shift();
                NetworkHandler_1.NetworkBus.emit(data.packet_id, data);
                NetworkHandler_1.NetworkChannelBus.emit(data.channel, data);
            }
        }
        setup() {
            (function (inst) {
                inst.logger.info('Starting up NetworkEngine.Client...');
                inst.socket = inst.io.connect('http://' + inst.config.ip + ':' + inst.config.port);
                inst.encrypt(global.ModLoader.version)(inst.socket);
                NetworkHandler_1.NetworkSendBus.addListener('msg', (data) => {
                    data.player = inst.me;
                    data.lobby = inst.config.lobby;
                    if (data.socketType === 1 /* UDP */ && inst.isUDPEnabled) {
                        inst.udpClient.send(JSON.stringify(data), inst.serverUDPPort, inst.config.ip);
                    }
                    else {
                        inst.socket.emit('msg', data);
                    }
                });
                NetworkHandler_1.NetworkSendBus.addListener('toPlayer', (data) => {
                    data.packet.player = inst.me;
                    data.packet.lobby = inst.config.lobby;
                    inst.socket.emit('toSpecificPlayer', data);
                });
                inst.socket.on('connect', () => {
                    inst.logger.info('Connected.');
                });
                inst.socket.on('uuid', (data) => {
                    inst.me = new ModLoaderDefaultImpls_1.NetworkPlayer(inst.config.nickname, data.uuid);
                    inst.socket.emit('version', global.ModLoader.version);
                });
                inst.socket.on('versionGood', (data) => {
                    inst.logger.info('Version good! ' + JSON.stringify(data.server));
                    let ld = new NetworkHandler_1.LobbyData(inst.config.lobby, crypto_1.default
                        .createHash('md5')
                        .update(Buffer.from(inst.config.password))
                        .digest('hex'));
                    EventHandler_1.bus.emit(EventHandler_1.EventsClient.CONFIGURE_LOBBY, ld);
                    if (inst.modLoaderconfig.patch !== '') {
                        ld.data['patch'] = zlib_1.default.gzipSync(fs_1.default.readFileSync(path_1.default.resolve(path_1.default.join('./mods', inst.modLoaderconfig.patch))));
                    }
                    inst.socket.emit('LobbyRequest', new LobbyJoin(ld, inst.me));
                    EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_SERVER_CONNECTION, {});
                });
                inst.socket.on('versionBad', (data) => {
                    inst.logger.info('Version bad! ' + JSON.stringify(data.server));
                });
                inst.socket.on('LobbyReady', (data) => {
                    let ld = data.storage;
                    let udpPort = data.udp;
                    inst.logger.info('Joined lobby ' + ld.name + '.');
                    let p = Buffer.alloc(1);
                    if (ld.data.hasOwnProperty('patch')) {
                        p = zlib_1.default.gunzipSync(ld.data.patch);
                    }
                    let udpTest = new UDPTestPacket();
                    udpTest.player = inst.me;
                    udpTest.lobby = inst.config.lobby;
                    inst.udpTestHandle = setTimeout(() => {
                        inst.isUDPEnabled = false;
                        inst.logger.error('UDP disabled.');
                    }, 30 * 1000);
                    inst.udpClient.send(JSON.stringify(udpTest), udpPort, inst.config.ip);
                    inst.serverUDPPort = udpPort;
                    modloader64_1.internal_event_bus.emit('onNetworkConnect', {
                        me: inst.me,
                        patch: p,
                    });
                    EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_LOBBY_JOIN, ld);
                });
                inst.socket.on('LobbyDenied_BadPassword', (ld) => {
                    inst.logger.error('Failed to join lobby. :(');
                });
                inst.socket.on('left', (player) => {
                    EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_PLAYER_LEAVE, player);
                });
                inst.socket.on('playerJoined', (player) => {
                    if (player.uuid !== inst.me.uuid) {
                        EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_PLAYER_JOIN, player);
                        inst.socket.emit('playerJoined_reply', {
                            player: inst.me,
                            dest: player,
                        });
                    }
                });
                inst.socket.on('playerJoined_bounce', (player) => {
                    if (player.uuid !== inst.me.uuid) {
                        EventHandler_1.bus.emit(EventHandler_1.EventsClient.ON_PLAYER_JOIN, player);
                    }
                });
                inst.socket.on('msg', (data) => {
                    inst.packetBuffer.push(data);
                });
                inst.socket.on('udpTest', (data) => {
                    inst.isUDPEnabled = true;
                    clearTimeout(inst.udpTestHandle);
                    inst.logger.info('UDP test passed.');
                });
            })(this);
        }
    }
    NetworkEngine.Client = Client;
})(NetworkEngine || (NetworkEngine = {}));
exports.default = NetworkEngine;
//# sourceMappingURL=NetworkEngine.js.map